import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { motion } from 'framer-motion'
import { Calendar, CheckCircle, Search, Sparkles, Zap, Smile, Heart } from 'lucide-react'
import './BookAppointment.css'

const BookAppointment = () => {
  const navigate = useNavigate()
  const [selectedService, setSelectedService] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [selectedClinic, setSelectedClinic] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const services = [
    { 
      id: 'examination', 
      name: 'Осмотр', 
      icon: Search, 
      color: '#10B981',
      description: 'Профилактический осмотр и консультация'
    },
    { 
      id: 'filling', 
      name: 'Пломба', 
      icon: Sparkles, 
      color: '#059669',
      description: 'Лечение кариеса и установка пломбы'
    },
    { 
      id: 'whitening', 
      name: 'Отбеливание', 
      icon: Zap, 
      color: '#34D399',
      description: 'Профессиональное отбеливание зубов'
    },
    { 
      id: 'implant', 
      name: 'Имплант', 
      icon: Heart, 
      color: '#047857',
      description: 'Установка зубного импланта'
    }
  ]

  const clinics = [
    'Стоматология "Улыбка"',
    'Клиника "ДентПро"',
    'Стоматология "Белый Клык"'
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedService && selectedDate && selectedClinic) {
      const savedVisits = localStorage.getItem('dentalVisits')
      const visits = savedVisits ? JSON.parse(savedVisits) : { active: [], past: [] }
      
      const service = services.find(s => s.id === selectedService)
      const newVisit = {
        id: Date.now(),
        clinic: selectedClinic,
        service: service.name,
        date: selectedDate,
        status: 'active'
      }

      visits.active.push(newVisit)
      localStorage.setItem('dentalVisits', JSON.stringify(visits))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedService('')
        setSelectedDate('')
        setSelectedClinic('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-appointment">
      <section className="page-intro">
        <motion.div
          className="intro-content"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <div className="intro-icon-wrapper">
            <Smile className="intro-main-icon" size={64} color="#10B981" />
          </div>
          <h1 className="intro-title">Записаться на прием</h1>
          <p className="intro-description">
            Выберите услугу, клинику и удобную дату для записи к стоматологу
          </p>
        </motion.div>
      </section>

      <div className="container">
        <section className="booking-section">
          <motion.form
            className="booking-form"
            onSubmit={handleSubmit}
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2, duration: 0.6 }}
          >
            <div className="form-group">
              <label className="form-label">
                <Search className="label-icon" size={20} color="#10B981" />
                Выберите услугу
              </label>
              <div className="services-grid">
                {services.map((service) => {
                  const IconComponent = service.icon
                  return (
                    <motion.button
                      key={service.id}
                      type="button"
                      className={`service-card ${selectedService === service.id ? 'selected' : ''}`}
                      onClick={() => setSelectedService(service.id)}
                      whileHover={{ scale: 1.05, y: -5 }}
                      whileTap={{ scale: 0.95 }}
                      style={{
                        borderColor: selectedService === service.id ? service.color : 'rgba(16, 185, 129, 0.2)',
                        background: selectedService === service.id 
                          ? `linear-gradient(135deg, ${service.color}20, ${service.color}10)`
                          : 'rgba(255, 255, 255, 0.6)'
                      }}
                    >
                      <div className="service-icon-wrapper" style={{ background: `${service.color}20` }}>
                        <IconComponent className="service-icon" size={32} color={service.color} />
                      </div>
                      <div className="service-content">
                        <h3 className="service-name">{service.name}</h3>
                        <p className="service-description">{service.description}</p>
                      </div>
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <div className="form-group">
              <label className="form-label">
                <Calendar className="label-icon" size={20} color="#10B981" />
                Выберите клинику
              </label>
              <select
                className="form-select"
                value={selectedClinic}
                onChange={(e) => setSelectedClinic(e.target.value)}
                required
              >
                <option value="">-- Выберите клинику --</option>
                {clinics.map((clinic) => (
                  <option key={clinic} value={clinic}>
                    {clinic}
                  </option>
                ))}
              </select>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar className="label-icon" size={20} color="#10B981" />
                Дата приема
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                min={minDate}
                max={maxDate}
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                required
              />
            </div>

            <motion.button
              type="submit"
              className="submit-btn"
              disabled={!selectedService || !selectedDate || !selectedClinic || isSubmitted}
              whileHover={{ scale: 1.02, y: -2 }}
              whileTap={{ scale: 0.98 }}
            >
              <CheckCircle className="btn-icon" size={20} color="white" />
              Записаться на прием
            </motion.button>

            {isSubmitted && (
              <motion.div
                className="success-message"
                initial={{ opacity: 0, y: -10 }}
                animate={{ opacity: 1, y: 0 }}
              >
                <CheckCircle className="success-icon" size={24} color="#10B981" />
                <span>Вы успешно записались на прием!</span>
              </motion.div>
            )}
          </motion.form>
        </section>

        <section className="info-section">
          <motion.div
            className="info-card"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.4, duration: 0.6 }}
          >
            <h3 className="info-title">Почему выбирают нас?</h3>
            <ul className="info-list">
              <li>Опытные врачи с многолетним стажем</li>
              <li>Современное оборудование и материалы</li>
              <li>Безболезненное лечение</li>
              <li>Гарантия на все виды работ</li>
            </ul>
          </motion.div>
        </section>
      </div>
    </div>
  )
}

export default BookAppointment

